<?php

namespace Modules\Awards\Awards;

use App\Contracts\Award;
use App\Models\UserAward;
use App\Models\UserField;
use App\Models\UserFieldValue;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class SPAwardsNetworkelite extends Award
{
    public $name = 'SPAwards(Networkelite)';
    
    public $param_description = 'The total combined flight time in minutes across both VATSIM and IVAO networks';

    public function check($totalMinutes = null): bool
    {
        // Load configuration
        $configPath = base_path('modules/Awards/spawards_config.php');
        if (!file_exists($configPath)) {
            Log::error('SPAwards(Networkelite) | Missing configuration file: modules/Awards/spawards_config.php');
            return false;
        }

        $config = include $configPath;

        $ivaoId  = $config['customfields']['ivao_id_field'] ?? null;
        $vatsimId  = $config['customfields']['vatsim_id_field'] ?? null;

        if (empty($ivaoId) || empty($vatsimId)) {
            Log::error('SPAwards(Networkelite) | Missing Networkelite configuration (VATSIM ID or IVAO ID).');
            return false;
        }
        
        // Ensure parameter is provided and valid
        if (is_null($totalMinutes) || !is_numeric($totalMinutes)) {
            Log::error('SPAwards(Networkelite) | Invalid or missing parameter.');
            return false;
        }

        // Check if the award is already granted
        $award = \App\Models\Award::where('ref_model', get_class($this))
            ->where('ref_model_params', (string) $totalMinutes)
            ->first();

        if (!$award) {
            Log::error("SPAwards(Networkelite) | No matching award found.");
            return false;
        }

        $alreadyGranted = UserAward::where('user_id', $this->user->id)
            ->where('award_id', $award->id)
            ->exists();

        if ($alreadyGranted) {
            Log::info("SPAwards(Networkelite) | Award already granted to Pilot (ID: {$this->user->id}). Skipping...");
            return false;
        }

        $requiredMinutes = (float) $totalMinutes;
        $totalNetworkMinutes = 0;

        // VATSIM TIME FETCH
        try {
            $vatsimFieldId = optional(
                UserField::select('id')->where('name', $config['customfields']['vatsim_id_field'])->first()
            )->id;

            if ($vatsimFieldId) {
                $vatsimId = UserFieldValue::where('user_field_id', $vatsimFieldId)
                    ->where('user_id', $this->user->id)
                    ->value('value');

                if (!empty($vatsimId)) {
                    $response = Http::acceptJson()->get("https://api.vatsim.net/v2/members/{$vatsimId}/stats");
                    if ($response->ok()) {
                        $data = $response->json();
                        if (isset($data['pilot'])) {
                            // Convert VATSIM hours to minutes
                            $vatsimMinutes = (float) $data['pilot'] * 60;
                            $totalNetworkMinutes += $vatsimMinutes;
                            Log::info("SPAwards(Networkelite) | VATSIM: {$vatsimMinutes} minutes for Pilot (ID: {$this->user->id}).");
                        }
                    }
                }
            }
        } catch (\Throwable $e) {
            Log::error("SPAwards(Networkelite) | Error fetching VATSIM data: {$e->getMessage()}");
        }

        // IVAO TIME FETCH
        try {
            $ivaoFieldId = optional(
                UserField::select('id')->where('name', $config['customfields']['ivao_id_field'])->first()
            )->id;

            if ($ivaoFieldId) {
                $ivaoId = UserFieldValue::where('user_field_id', $ivaoFieldId)
                    ->where('user_id', $this->user->id)
                    ->value('value');

                if (!empty($ivaoId)) {
                    $response = Http::withHeaders([
                        'accept' => 'application/json',
                        'apiKey' => $config['ivao']['api_key'],
                    ])->get("https://api.ivao.aero/v2/users/{$ivaoId}");

                    if ($response->ok()) {
                        $data = $response->json();
                        $pilotEntry = collect($data['hours'] ?? [])->firstWhere('type', 'pilot');
                        if ($pilotEntry && isset($pilotEntry['hours'])) {
                            // Convert IVAO seconds to minutes
                            $ivaoMinutes = round(((float) $pilotEntry['hours']) / 60, 1);
                            $totalNetworkMinutes += $ivaoMinutes;
                            Log::info("SPAwards(Networkelite) | IVAO: {$ivaoMinutes} minutes for Pilot (ID: {$this->user->id}).");
                        }
                    }
                }
            }
        } catch (\Throwable $e) {
            Log::error("SPAwards(Networkelite) | Error fetching IVAO data: {$e->getMessage()}");
        }

        // Log for debugging
        Log::info("SPAwards(Networkelite) | Pilot (ID: {$this->user->id}) total combined minutes: {$totalNetworkMinutes}, required: {$requiredMinutes}.");

        // Return true if the total meets or exceeds the requirement
        return $totalNetworkMinutes >= $requiredMinutes;
    }
}
